set more off 
pause off
set logtype text
set mem 500M

*************** DESCRIPTION *************************************
* Computes Modified-HHIs at the industry-level using ownership 
* data from Thomson Reuters 13f. 
*
*	Inputs: CRSP --> for MV and mapping to GVKEY
*			Bushee_detailed --> Ownership data
*			data_firm --> mapping of GVKEY to indcode
*
*	Outputs: MHHI by indcode, from 1981 onward
*****************************************************************

forval yy = 1980/2015{

* Prepare crsp dataset
use 0.raw_inputs\crsp_data, clear
g cusip8 = substr(ncusip, 1, 8)
g year = year(date)
keep if month(date)== 12
g mktcap = abs(prc) * shrout

* Some ncusips are not populated. We could replace them with the most recent cusip 
* including the following code, but it would not properly replicate history 
* replace cusip8 = substr(cusip,1,8) if cusip8 == "" 
drop if cusip8 == "" 
save temp, replace

***

/* ---------------------------------- */
/* Get CRSP permco, shrout and price  */
/* ---------------------------------- */
use 2.intermediate\bushee_detailed, clear
rename cusip cusip8
keep if year == `yy'

* Keep only if ownership share exceeds 0.5%
keep mgrno_mapped cusip8 year shares sole shared no
rename mgrno_mapped mgrno
collapse (sum) shares sole shared no, by(year cusip8 mgrno )
egen temp = sum(shares), by(cusip8)
g ownshare = shares/temp 
drop if ownshare < 0.005
drop temp ownshare

* Recompute share for firms with more than 0.5% ownership
egen temp = sum(shares), by(cusip8)
g ownshare = shares/temp 
drop temp

egen temp = sum(sole + shared), by(cusip8)
g votshare = (sole + shared)/temp 
drop temp

merge m:1 cusip8 year using temp, keepusing(permco mktcap )
drop if _merge ~= 3
drop _merge

* In case of multiple cusip8 (i.e., multiple permno) for a given permco, 
* select cusip8 with largest market capitalization
g negmktcap = -mktcap
sort permco mgrno year negmktcap
bys permco mgrno year : keep if _n==1
save temp, replace

***

/* ------------*/
/* Grab Gvkey  */
/* ------------*/

* join bushee dataset with CCM table to obtain GVKEY for chosen permno
use 0.raw_inputs\linkingTable, clear
rename lpermco permco
joinby permco using temp, unmatched(none)

* Keep only if gvkey is relevant for chosen years
keep if year(linkdt) <= year & year(linkenddt) >= year

* drop duplicates
sort gvkey year mgrno negmktcap
bys gvkey year mgrno : keep if _n==1
drop negmktcap

destring, replace
save temp, replace

* Merge indcode 
use 2.intermediate\data_firm, clear
drop if sale == .

bys gvkey year: keep if _n == 1
drop if gvkey == . | indcode == ""
merge 1:m gvkey year using temp

keep if mgrno ~= . 
keep gvkey year indcode mgrno ownshare votshare  ss1 
drop if indcode == ""

* Compute common ownership adjustment
egen a1_ownden = sum(ownshare^2), by(gvkey)
egen a1_votden = sum(votshare^2), by(gvkey)

sort indcode year mgrno gvkey
by indcode year mgrno: gen gfreq = _N
expand gfreq
sort indcode year mgrno gvkey
by indcode year mgrno gvkey: gen numid2 = _n
by indcode year mgrno: gen gvkey2 = gvkey[gfreq * numid2]
by indcode year mgrno: gen ss2 = ss1[gfreq * numid2]
by indcode year mgrno: gen ownshare2 = ownshare[gfreq * numid2]
by indcode year mgrno: gen votshare2 = votshare[gfreq * numid2]
drop if gvkey == gvkey2

g ownprod = ownshare*ownshare2
g votprod = votshare*votshare2
egen a1_ownnum = sum(ownprod), by(gvkey gvkey2)
egen a1_votnum = sum(votprod), by(gvkey gvkey2)
g ssprod = ss1*ss2

sort gvkey gvkey2 
bys gvkey gvkey2 : keep if _n ==1
egen herf_mod = sum(ssprod*a1_ownnum / a1_ownden), by(indcode year)
egen herf_votmod = sum(ssprod*a1_votnum / a1_votden), by(indcode year)

keep year indcode herf_mod herf_votmod
bys year indcode: keep if _n == 1 

save 2.intermediate\modHHI\mod_HHI_`yy'_$currseg, replace
}


forval yy = 1980/2015{

use 2.intermediate\modHHI\mod_HHI_`yy'_$currseg, clear
if `yy' ~= 1980 {
	append using temp
}
save temp, replace
sleep 600
}

replace herf_mod = 0 if herf_mod<0 & herf_mod~=.
replace herf_mod = 1 if herf_mod>1 & herf_mod~=.

replace herf_mod = 0 if herf_mod<0 & herf_mod~=.
replace herf_mod = 1 if herf_mod>1 & herf_mod~=.

save 2.intermediate\mod_HHI_$currseg, replace
